<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class GenerateVAPIDKeys extends Command
{
    protected $signature = 'pwa:generate-vapid';
    protected $description = 'Genera claves VAPID para push notifications';

    public function handle()
    {
        $this->info('Generando claves VAPID...');
        
        try {
            // Intentar usar la librería minishlink/web-push si está disponible
            if (class_exists('\Minishlink\WebPush\VAPID')) {
                try {
                    $keys = \Minishlink\WebPush\VAPID::createVapidKeys();
                    $this->displayKeys($keys);
                    return Command::SUCCESS;
                } catch (\Exception $e) {
                    $this->warn('Error con minishlink/web-push: ' . $e->getMessage());
                    $this->info('Intentando método alternativo...');
                }
            }
            
            // Si no está disponible, usar método alternativo
            $this->info('Generando claves usando método alternativo...');
            
            $keys = $this->generateVAPIDKeysAlternative();
            $this->displayKeys($keys);
            
            return Command::SUCCESS;
        } catch (\Exception $e) {
            $this->error('Error generando claves VAPID: ' . $e->getMessage());
            $this->info('');
            $this->info('═══════════════════════════════════════════════════════════');
            $this->info('Alternativas para generar claves VAPID:');
            $this->info('═══════════════════════════════════════════════════════════');
            $this->info('');
            $this->info('1. Usar Node.js (Recomendado):');
            $this->info('   npm install -g web-push');
            $this->info('   web-push generate-vapid-keys');
            $this->info('');
            $this->info('2. Usar herramienta online:');
            $this->info('   https://web-push-codelab.glitch.me/');
            $this->info('   https://www.stephane-quantin.com/en/tools/generators/vapid-keys');
            $this->info('');
            $this->info('3. Verificar instalación de minishlink/web-push:');
            $this->info('   composer require minishlink/web-push');
            $this->info('   composer dump-autoload');
            $this->info('');
            
            return Command::FAILURE;
        }
    }
    
    private function generateVAPIDKeysAlternative()
    {
        // Generar claves usando OpenSSL
        if (!extension_loaded('openssl')) {
            throw new \Exception('La extensión OpenSSL no está disponible');
        }
        
        $config = [
            'digest_alg' => 'sha256',
            'private_key_bits' => 2048,
            'private_key_type' => OPENSSL_KEYTYPE_EC,
            'curve_name' => 'prime256v1',
        ];
        
        $resource = openssl_pkey_new($config);
        if (!$resource) {
            throw new \Exception('Error generando clave: ' . openssl_error_string());
        }
        
        openssl_pkey_export($resource, $privateKey);
        $publicKey = openssl_pkey_get_details($resource)['key'];
        
        // Convertir a formato VAPID (PEM a Base64 URL-safe)
        $publicKeyPEM = openssl_pkey_get_public($publicKey);
        $publicKeyDetails = openssl_pkey_get_details($publicKeyPEM);
        
        // Extraer coordenadas del punto público
        $x = bin2hex($publicKeyDetails['ec']['x']);
        $y = bin2hex($publicKeyDetails['ec']['y']);
        
        // Convertir a Base64 URL-safe
        $publicKeyBase64 = $this->base64UrlEncode(hex2bin('04' . $x . $y));
        $privateKeyBase64 = $this->base64UrlEncode($this->extractPrivateKeyBytes($privateKey));
        
        return [
            'publicKey' => $publicKeyBase64,
            'privateKey' => $privateKeyBase64
        ];
    }
    
    private function extractPrivateKeyBytes($privateKeyPEM)
    {
        // Extraer bytes de la clave privada
        $privateKeyResource = openssl_pkey_get_private($privateKeyPEM);
        $details = openssl_pkey_get_details($privateKeyResource);
        return $details['ec']['d'];
    }
    
    private function base64UrlEncode($data)
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    
    private function displayKeys($keys)
    {
        $this->info('');
        $this->info('✅ Claves VAPID generadas exitosamente!');
        $this->info('');
        $this->info('═══════════════════════════════════════════════════════════');
        $this->info('Agrega estas líneas a tu archivo .env:');
        $this->info('═══════════════════════════════════════════════════════════');
        $this->info('');
        $this->line('VAPID_SUBJECT=mailto:tu-email@ejemplo.com');
        $this->line('VAPID_PUBLIC_KEY=' . $keys['publicKey']);
        $this->line('VAPID_PRIVATE_KEY=' . $keys['privateKey']);
        $this->info('');
        $this->warn('⚠️  IMPORTANTE:');
        $this->warn('   - Reemplaza "tu-email@ejemplo.com" con un email válido');
        $this->warn('   - Nunca compartas tu clave privada');
        $this->warn('   - Guarda estas claves de forma segura');
        $this->warn('   - No las subas a repositorios públicos');
        $this->info('');
        $this->info('Después de agregar las claves, ejecuta:');
        $this->info('  php artisan config:clear');
        $this->info('');
    }
}
