<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cliente;
use App\Models\User;
use App\Services\ClienteService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ClienteController extends Controller
{
    protected $clienteService;

    public function __construct(ClienteService $clienteService)
    {
        $this->clienteService = $clienteService;
    }

    public function index(Request $request)
    {
        $query = Cliente::with(['vendedor', 'documentos']);

        // Búsqueda por nombre de empresa
        if ($request->filled('buscar')) {
            $buscar = $request->get('buscar');
            $query->where(function($q) use ($buscar) {
                $q->where('nombre_empresa', 'like', "%{$buscar}%")
                  ->orWhere('rif', 'like', "%{$buscar}%");
            });
        }

        // Filtro por vendedor
        if ($request->filled('vendedor_id')) {
            $query->where('vendedor_id', $request->get('vendedor_id'));
        }

        // Filtro por zona
        if ($request->filled('zona')) {
            $query->where('zona', $request->get('zona'));
        }

        // Ordenamiento
        $orden = $request->get('orden', 'nombre_empresa');
        $direccion = $request->get('direccion', 'asc');
        
        if ($orden !== 'estatus') {
            $query->orderBy($orden, $direccion);
        } else {
            $query->orderBy('nombre_empresa', 'asc');
        }

        $perPage = $request->get('per_page', 20);
        $clientes = $query->get();

        // Aplicar filtro de estatus si existe
        if ($request->filled('estatus')) {
            $estatusFiltro = $request->get('estatus');
            $clientes = $clientes->filter(function($cliente) use ($estatusFiltro) {
                $estatus = $cliente->calcularEstatus();
                return $estatus['texto'] === $estatusFiltro;
            });
        }

        // Ordenar por estatus si es necesario
        if ($orden === 'estatus') {
            $ordenEstatus = [
                'VENCIDO' => 1,
                'ALERTA' => 2,
                'SIN DOCUMENTOS' => 3,
                'ACTUALIZADO' => 4
            ];
            $clientes = $clientes->sortBy(function($cliente) use ($ordenEstatus) {
                $estatus = $cliente->calcularEstatus();
                return $ordenEstatus[$estatus['texto']] ?? 5;
            }, SORT_REGULAR, $direccion === 'desc');
        }

        // Paginar manualmente
        $currentPage = \Illuminate\Pagination\LengthAwarePaginator::resolveCurrentPage();
        $items = $clientes->slice(($currentPage - 1) * $perPage, $perPage)->values();
        $clientes = new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $clientes->count(),
            $perPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        $vendedores = \App\Models\User::vendedores()->activos()->get();
        $zonas = Cliente::distinct()->whereNotNull('zona')->pluck('zona')->sort();

        return view('admin.clientes.index', compact('clientes', 'vendedores', 'zonas'));
    }

    public function create()
    {
        $vendedores = User::vendedores()->activos()->get();
        return view('admin.clientes.create', compact('vendedores'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre_empresa' => 'required|string|max:255',
            'rif' => 'required|string|max:50|unique:clientes,rif',
            'direccion_fiscal' => 'nullable|string',
            'zona' => 'nullable|string|max:100',
            'vendedor_id' => 'nullable|exists:users,id',
            'notas' => 'nullable|string',
        ]);

        $cliente = $this->clienteService->crearCliente($validated, auth()->id());

        // Crear directorio para archivos
        $rutaCarpeta = 'archivos/' . $cliente->rif;
        Storage::disk('public')->makeDirectory($rutaCarpeta);

        return redirect()->route('admin.clientes.index')
            ->with('success', 'Cliente creado exitosamente.');
    }

    public function show(Cliente $cliente)
    {
        $cliente->load(['vendedor', 'documentos.tipoDocumento', 'historialCambios.usuario']);
        $estatus = $cliente->calcularEstatus();
        
        return view('admin.clientes.show', compact('cliente', 'estatus'));
    }

    public function edit(Cliente $cliente)
    {
        $vendedores = User::vendedores()->activos()->get();
        return view('admin.clientes.edit', compact('cliente', 'vendedores'));
    }

    public function update(Request $request, Cliente $cliente)
    {
        $validated = $request->validate([
            'nombre_empresa' => 'required|string|max:255',
            'rif' => 'required|string|max:50|unique:clientes,rif,' . $cliente->id,
            'direccion_fiscal' => 'nullable|string',
            'zona' => 'nullable|string|max:100',
            'vendedor_id' => 'nullable|exists:users,id',
            'notas' => 'nullable|string',
        ]);

        $this->clienteService->actualizarCliente($cliente, $validated, auth()->id());

        return redirect()->route('admin.clientes.index')
            ->with('success', 'Cliente actualizado exitosamente.');
    }

    public function destroy(Cliente $cliente)
    {
        // Eliminar archivos del cliente
        $rutaCarpeta = 'archivos/' . $cliente->rif;
        Storage::disk('public')->deleteDirectory($rutaCarpeta);

        $cliente->delete();

        return redirect()->route('admin.clientes.index')
            ->with('success', 'Cliente eliminado exitosamente.');
    }
}

