<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Configuracion;
use Illuminate\Http\Request;

class ConfiguracionController extends Controller
{
    public function index()
    {
        $configuraciones = Configuracion::orderBy('categoria')
            ->orderBy('clave')
            ->get()
            ->groupBy('categoria');

        // Valores por defecto si no existen
        $defaults = [
            'documentos' => [
                'dias_alerta_vencimiento' => ['valor' => 180, 'descripcion' => 'Días de anticipación para alerta de vencimiento (por defecto: 6 meses)'],
                'tamaño_maximo_archivo_mb' => ['valor' => 10, 'descripcion' => 'Tamaño máximo de archivo en MB'],
                'tipos_archivo_permitidos' => ['valor' => 'pdf,jpg,jpeg,png', 'descripcion' => 'Tipos de archivo permitidos (separados por coma)'],
            ],
            'sistema' => [
                'items_por_pagina' => ['valor' => 20, 'descripcion' => 'Número de items por página por defecto'],
                'actualizacion_notificaciones_segundos' => ['valor' => 30, 'descripcion' => 'Intervalo de actualización de notificaciones en segundos'],
            ],
            'apariencia' => [
                'color_primario' => ['valor' => '#0d6efd', 'descripcion' => 'Color primario del sistema'],
                'color_secundario' => ['valor' => '#6c757d', 'descripcion' => 'Color secundario del sistema'],
            ],
        ];

        // Crear configuraciones por defecto si no existen
        foreach ($defaults as $categoria => $configs) {
            foreach ($configs as $clave => $data) {
                if (!Configuracion::where('clave', $clave)->exists()) {
                    Configuracion::establecer(
                        $clave,
                        $data['valor'],
                        is_numeric($data['valor']) ? 'number' : 'text',
                        $data['descripcion'],
                        $categoria
                    );
                }
            }
        }

        // Recargar configuraciones
        $configuraciones = Configuracion::orderBy('categoria')
            ->orderBy('clave')
            ->get()
            ->groupBy('categoria');

        return view('admin.configuracion.index', compact('configuraciones'));
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            'configuraciones' => 'required|array',
        ]);

        foreach ($validated['configuraciones'] as $clave => $valor) {
            $config = Configuracion::where('clave', $clave)->first();
            
            if ($config) {
                // Convertir según el tipo
                $valorFinal = match($config->tipo) {
                    'boolean' => $valor ? '1' : '0',
                    'number' => (string)$valor,
                    'json' => is_array($valor) ? json_encode($valor) : $valor,
                    default => $valor,
                };

                $config->update(['valor' => $valorFinal]);
            }
        }

        return redirect()->back()
            ->with('success', 'Configuraciones actualizadas exitosamente.');
    }
}

