<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cliente;
use App\Models\Documento;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;

class ExportController extends Controller
{
    public function exportarClientes(Request $request)
    {
        $query = Cliente::with(['vendedor', 'documentos']);

        // Aplicar mismos filtros que en index
        if ($request->filled('buscar')) {
            $buscar = $request->get('buscar');
            $query->where(function($q) use ($buscar) {
                $q->where('nombre_empresa', 'like', "%{$buscar}%")
                  ->orWhere('rif', 'like', "%{$buscar}%");
            });
        }

        if ($request->filled('vendedor_id')) {
            $query->where('vendedor_id', $request->get('vendedor_id'));
        }

        if ($request->filled('zona')) {
            $query->where('zona', $request->get('zona'));
        }

        $clientes = $query->orderBy('nombre_empresa')->get();

        // Generar CSV
        $filename = 'clientes_' . date('Y-m-d_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($clientes) {
            $file = fopen('php://output', 'w');
            
            // BOM para Excel
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // Encabezados
            fputcsv($file, [
                'Nombre Empresa',
                'RIF',
                'Dirección Fiscal',
                'Zona',
                'Vendedor',
                'Estatus',
                'Total Documentos',
                'Documentos Vencidos',
                'Fecha Creación'
            ], ';');

            // Datos
            foreach ($clientes as $cliente) {
                $estatus = $cliente->calcularEstatus();
                $documentosVencidos = $cliente->documentos()
                    ->whereNotNull('fecha_vencimiento')
                    ->where('fecha_vencimiento', '<', now())
                    ->count();

                fputcsv($file, [
                    $cliente->nombre_empresa,
                    $cliente->rif,
                    $cliente->direccion_fiscal ?? '',
                    $cliente->zona ?? '',
                    $cliente->vendedor ? $cliente->vendedor->nombre : 'Sin asignar',
                    $estatus['texto'],
                    $cliente->documentos->count(),
                    $documentosVencidos,
                    $cliente->created_at->format('d/m/Y H:i')
                ], ';');
            }

            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }

    public function exportarDocumentosVencidos()
    {
        $documentos = Documento::with(['cliente', 'tipoDocumento'])
            ->whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '<', now())
            ->orderBy('fecha_vencimiento')
            ->get();

        $filename = 'documentos_vencidos_' . date('Y-m-d_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($documentos) {
            $file = fopen('php://output', 'w');
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            fputcsv($file, [
                'Cliente',
                'RIF',
                'Tipo de Documento',
                'Nombre Archivo',
                'Fecha de Vencimiento',
                'Días Vencido',
                'Vendedor'
            ], ';');

            foreach ($documentos as $doc) {
                $diasVencido = now()->diffInDays($doc->fecha_vencimiento);
                
                fputcsv($file, [
                    $doc->cliente->nombre_empresa,
                    $doc->cliente->rif,
                    $doc->tipoDocumento->nombre,
                    $doc->nombre_archivo,
                    $doc->fecha_vencimiento->format('d/m/Y'),
                    $diasVencido,
                    $doc->cliente->vendedor ? $doc->cliente->vendedor->nombre : 'Sin asignar'
                ], ';');
            }

            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }

    public function exportarDocumentosProximosAVencer()
    {
        $documentos = Documento::with(['cliente', 'tipoDocumento'])
            ->whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '>=', now())
            ->where('fecha_vencimiento', '<=', now()->addMonths(6))
            ->orderBy('fecha_vencimiento')
            ->get();

        $filename = 'documentos_proximos_vencer_' . date('Y-m-d_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($documentos) {
            $file = fopen('php://output', 'w');
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            fputcsv($file, [
                'Cliente',
                'RIF',
                'Tipo de Documento',
                'Nombre Archivo',
                'Fecha de Vencimiento',
                'Días Restantes',
                'Vendedor'
            ], ';');

            foreach ($documentos as $doc) {
                $diasRestantes = now()->diffInDays($doc->fecha_vencimiento, false);
                
                fputcsv($file, [
                    $doc->cliente->nombre_empresa,
                    $doc->cliente->rif,
                    $doc->tipoDocumento->nombre,
                    $doc->nombre_archivo,
                    $doc->fecha_vencimiento->format('d/m/Y'),
                    $diasRestantes,
                    $doc->cliente->vendedor ? $doc->cliente->vendedor->nombre : 'Sin asignar'
                ], ';');
            }

            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }

    public function exportarDocumentosCliente(Cliente $cliente)
    {
        $documentos = $cliente->documentos()->with('tipoDocumento')->get();

        $filename = 'documentos_' . str_replace([' ', '/'], '_', $cliente->nombre_empresa) . '_' . date('Y-m-d_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($documentos, $cliente) {
            $file = fopen('php://output', 'w');
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            fputcsv($file, [
                'Cliente: ' . $cliente->nombre_empresa,
                'RIF: ' . $cliente->rif
            ], ';');
            fputcsv($file, []); // Línea vacía
            
            fputcsv($file, [
                'Tipo de Documento',
                'Nombre Archivo',
                'Fecha de Vencimiento',
                'Estado',
                'Observaciones',
                'Fecha de Subida'
            ], ';');

            foreach ($documentos as $doc) {
                $estado = 'Sin fecha';
                if ($doc->fecha_vencimiento) {
                    if ($doc->esta_vencido) {
                        $estado = 'Vencido';
                    } elseif ($doc->proximo_a_vencer) {
                        $estado = 'Próximo a vencer';
                    } else {
                        $estado = 'Vigente';
                    }
                }

                fputcsv($file, [
                    $doc->tipoDocumento->nombre,
                    $doc->nombre_archivo,
                    $doc->fecha_vencimiento ? $doc->fecha_vencimiento->format('d/m/Y') : '',
                    $estado,
                    $doc->observaciones ?? '',
                    $doc->created_at->format('d/m/Y H:i')
                ], ';');
            }

            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }
}

