<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Notificacion;
use App\Services\PushNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class NotificacionController extends Controller
{
    protected $pushService;

    public function __construct(PushNotificationService $pushService)
    {
        $this->pushService = $pushService;
    }

    public function index()
    {
        $vendedores = User::vendedores()->activos()->get();
        $notificacionesRecientes = Notificacion::with('destinatario')
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        return view('admin.notificaciones.index', compact('vendedores', 'notificacionesRecientes'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'destinatario_id' => 'required|exists:users,id',
            'titulo' => 'required|string|max:255',
            'mensaje' => 'required|string',
            'tipo' => 'nullable|string|max:50',
            'enviar_push' => 'nullable|boolean',
        ]);

        DB::beginTransaction();
        try {
            // Crear notificación en base de datos
            $notificacion = Notificacion::create([
                'destinatario_id' => $validated['destinatario_id'],
                'titulo' => $validated['titulo'],
                'mensaje' => $validated['mensaje'],
                'tipo' => $validated['tipo'] ?? 'general',
                'leida' => false,
            ]);

            // Enviar push notification si está habilitado
            if ($request->has('enviar_push') && $request->enviar_push) {
                $usuario = User::find($validated['destinatario_id']);
                $resultado = $this->pushService->enviarNotificacion(
                    $usuario,
                    $validated['titulo'],
                    $validated['mensaje'],
                    ['notificacion_id' => $notificacion->id]
                );
            }

            DB::commit();

            return redirect()->route('admin.notificaciones.index')
                ->with('success', 'Notificación enviada exitosamente.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error al enviar la notificación: ' . $e->getMessage());
        }
    }

    public function enviarATodos(Request $request)
    {
        $validated = $request->validate([
            'titulo' => 'required|string|max:255',
            'mensaje' => 'required|string',
            'tipo' => 'nullable|string|max:50',
            'enviar_push' => 'nullable|boolean',
        ]);

        DB::beginTransaction();
        try {
            $vendedores = User::vendedores()->activos()->get();
            $enviadas = 0;

            foreach ($vendedores as $vendedor) {
                Notificacion::create([
                    'destinatario_id' => $vendedor->id,
                    'titulo' => $validated['titulo'],
                    'mensaje' => $validated['mensaje'],
                    'tipo' => $validated['tipo'] ?? 'general',
                    'leida' => false,
                ]);
                $enviadas++;
            }

            // Enviar push a todos si está habilitado
            if ($request->has('enviar_push') && $request->enviar_push) {
                $this->pushService->enviarATodos(
                    $validated['titulo'],
                    $validated['mensaje'],
                    ['tipo' => $validated['tipo'] ?? 'general']
                );
            }

            DB::commit();

            return redirect()->route('admin.notificaciones.index')
                ->with('success', "Notificación enviada a {$enviadas} vendedores exitosamente.");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error al enviar las notificaciones: ' . $e->getMessage());
        }
    }

    public function marcarLeida(Notificacion $notificacion)
    {
        // Verificar que la notificación pertenece al usuario autenticado
        if ($notificacion->destinatario_id !== auth()->id()) {
            abort(403, 'No tienes acceso a esta notificación.');
        }

        $notificacion->update(['leida' => true]);

        return response()->json(['success' => true]);
    }

    public function marcarTodasLeidas()
    {
        Notificacion::where('destinatario_id', auth()->id())
            ->where('leida', false)
            ->update(['leida' => true]);

        return response()->json(['success' => true]);
    }

    public function obtenerNoLeidas()
    {
        $notificaciones = Notificacion::where('destinatario_id', auth()->id())
            ->noLeidas()
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'notificaciones' => $notificaciones->map(function($notif) {
                return [
                    'id' => $notif->id,
                    'titulo' => $notif->titulo,
                    'mensaje' => $notif->mensaje,
                    'tipo' => $notif->tipo,
                    'fecha' => $notif->created_at->diffForHumans(),
                ];
            }),
            'total' => Notificacion::where('destinatario_id', auth()->id())->noLeidas()->count()
        ]);
    }
}

