<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cliente;
use App\Models\Documento;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReporteController extends Controller
{
    public function index()
    {
        return view('admin.reportes.index');
    }

    public function documentosVencidosPorVendedor(Request $request)
    {
        $vendedorId = $request->get('vendedor_id');
        
        $query = Documento::with(['cliente.vendedor', 'tipoDocumento'])
            ->whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '<', now());

        if ($vendedorId) {
            $query->whereHas('cliente', function($q) use ($vendedorId) {
                $q->where('vendedor_id', $vendedorId);
            });
        }

        $documentos = $query->orderBy('fecha_vencimiento')->get();

        // Agrupar por vendedor
        $porVendedor = $documentos->groupBy(function($doc) {
            return $doc->cliente->vendedor ? $doc->cliente->vendedor->id : 'sin_vendedor';
        });

        $vendedores = User::vendedores()->activos()->get();

        return view('admin.reportes.documentos-vencidos-vendedor', [
            'porVendedor' => $porVendedor,
            'vendedores' => $vendedores,
            'vendedorSeleccionado' => $vendedorId,
            'total' => $documentos->count(),
        ]);
    }

    public function documentosProximosAVencer(Request $request)
    {
        $meses = $request->get('meses', 6);
        
        $documentos = Documento::with(['cliente.vendedor', 'tipoDocumento'])
            ->whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '>=', now())
            ->where('fecha_vencimiento', '<=', now()->addMonths($meses))
            ->orderBy('fecha_vencimiento')
            ->get();

        // Agrupar por mes de vencimiento
        $porMes = $documentos->groupBy(function($doc) {
            return $doc->fecha_vencimiento->format('Y-m');
        });

        return view('admin.reportes.documentos-proximos-vencer', [
            'porMes' => $porMes,
            'meses' => $meses,
            'total' => $documentos->count(),
        ]);
    }

    public function actividadVendedores(Request $request)
    {
        $fechaInicio = $request->get('fecha_inicio', now()->subMonth()->format('Y-m-d'));
        $fechaFin = $request->get('fecha_fin', now()->format('Y-m-d'));

        $vendedores = User::vendedores()->activos()->get();

        $actividades = [];
        foreach ($vendedores as $vendedor) {
            $clientes = $vendedor->clientes()->count();
            $documentosSubidos = Documento::whereHas('cliente', function($q) use ($vendedor) {
                $q->where('vendedor_id', $vendedor->id);
            })
            ->whereBetween('created_at', [$fechaInicio, $fechaFin])
            ->count();

            $solicitudesCreacion = \App\Models\SolicitudCreacion::where('vendedor_id', $vendedor->id)
                ->whereBetween('fecha_solicitud', [$fechaInicio, $fechaFin])
                ->count();

            $solicitudesEdicion = \App\Models\SolicitudEdicion::where('vendedor_id', $vendedor->id)
                ->whereBetween('fecha_solicitud', [$fechaInicio, $fechaFin])
                ->count();

            $actividades[] = [
                'vendedor' => $vendedor,
                'clientes' => $clientes,
                'documentos_subidos' => $documentosSubidos,
                'solicitudes_creacion' => $solicitudesCreacion,
                'solicitudes_edicion' => $solicitudesEdicion,
            ];
        }

        return view('admin.reportes.actividad-vendedores', [
            'actividades' => $actividades,
            'fechaInicio' => $fechaInicio,
            'fechaFin' => $fechaFin,
        ]);
    }

    public function clientesSinDocumentos()
    {
        $clientes = Cliente::with(['vendedor', 'documentos'])
            ->get()
            ->filter(function($cliente) {
                return $cliente->documentos->isEmpty();
            });

        return view('admin.reportes.clientes-sin-documentos', [
            'clientes' => $clientes,
            'total' => $clientes->count(),
        ]);
    }
}

