<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TipoDocumento;
use Illuminate\Http\Request;

class TipoDocumentoController extends Controller
{
    public function index()
    {
        $tiposDocumento = TipoDocumento::orderBy('nombre')->paginate(20);
        return view('admin.tipos-documento.index', compact('tiposDocumento'));
    }

    public function create()
    {
        return view('admin.tipos-documento.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => 'required|string|max:255|unique:tipos_documento,nombre',
            'descripcion' => 'nullable|string',
            'activo' => 'nullable|boolean',
        ]);

        TipoDocumento::create([
            'nombre' => $validated['nombre'],
            'descripcion' => $validated['descripcion'] ?? null,
            'activo' => $request->has('activo') ? true : false,
        ]);

        return redirect()->route('admin.tipos-documento.index')
            ->with('success', 'Tipo de documento creado exitosamente.');
    }

    public function edit(TipoDocumento $tiposDocumento)
    {
        return view('admin.tipos-documento.edit', compact('tiposDocumento'));
    }

    public function update(Request $request, TipoDocumento $tiposDocumento)
    {
        $validated = $request->validate([
            'nombre' => 'required|string|max:255|unique:tipos_documento,nombre,' . $tiposDocumento->id,
            'descripcion' => 'nullable|string',
            'activo' => 'nullable|boolean',
        ]);

        $tiposDocumento->update([
            'nombre' => $validated['nombre'],
            'descripcion' => $validated['descripcion'] ?? null,
            'activo' => $request->has('activo') ? true : false,
        ]);

        return redirect()->route('admin.tipos-documento.index')
            ->with('success', 'Tipo de documento actualizado exitosamente.');
    }

    public function destroy(TipoDocumento $tiposDocumento)
    {
        // Verificar si tiene documentos asociados
        if ($tiposDocumento->documentos()->count() > 0) {
            return redirect()->route('admin.tipos-documento.index')
                ->with('error', 'No se puede eliminar este tipo de documento porque tiene documentos asociados.');
        }

        $tiposDocumento->delete();

        return redirect()->route('admin.tipos-documento.index')
            ->with('success', 'Tipo de documento eliminado exitosamente.');
    }
}

