<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PushSubscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class PushNotificationController extends Controller
{
    public function getPublicKey()
    {
        try {
            $publicKey = config('services.vapid.public_key') ?? env('VAPID_PUBLIC_KEY');
            
            if (empty($publicKey)) {
                return response()->json([
                    'error' => 'VAPID public key not configured',
                    'publicKey' => null
                ], 200); // Retornar 200 pero con error en el JSON para que no falle el fetch
            }
            
            return response()->json([
                'publicKey' => $publicKey
            ], 200, [
                'Content-Type' => 'application/json',
                'Cache-Control' => 'no-cache, no-store, must-revalidate'
            ]);
        } catch (\Exception $e) {
            Log::error('Error obteniendo VAPID public key: ' . $e->getMessage());
            return response()->json([
                'error' => 'Error retrieving public key',
                'publicKey' => null
            ], 200);
        }
    }

    public function subscribe(Request $request)
    {
        $validated = $request->validate([
            'endpoint' => 'required|url',
            'keys' => 'required|array',
            'keys.p256dh' => 'required|string',
            'keys.auth' => 'required|string',
        ]);

        $user = Auth::user();

        PushSubscription::updateOrCreate(
            [
                'user_id' => $user->id,
                'endpoint' => $validated['endpoint'],
            ],
            [
                'p256dh' => $validated['keys']['p256dh'],
                'auth' => $validated['keys']['auth'],
                'user_agent' => $request->userAgent(),
            ]
        );

        return response()->json(['success' => true]);
    }

    public function unsubscribe(Request $request)
    {
        $validated = $request->validate([
            'endpoint' => 'required|url',
        ]);

        $user = Auth::user();

        PushSubscription::where('user_id', $user->id)
            ->where('endpoint', $validated['endpoint'])
            ->delete();

        return response()->json(['success' => true]);
    }

    public function verifySubscription(Request $request)
    {
        $validated = $request->validate([
            'endpoint' => 'required|url',
        ]);

        $user = Auth::user();

        $subscription = PushSubscription::where('user_id', $user->id)
            ->where('endpoint', $validated['endpoint'])
            ->first();

        return response()->json([
            'valid' => $subscription !== null
        ]);
    }
}

