<?php

namespace App\Http\Controllers\Vendedor;

use App\Http\Controllers\Controller;
use App\Models\Cliente;
use App\Models\Documento;
use App\Models\TipoDocumento;
use App\Models\HistorialCambio;
use App\Services\FileStorageService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class DocumentoController extends Controller
{
    protected $fileService;

    public function __construct(FileStorageService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'cliente_id' => 'required|exists:clientes,id',
            'tipo_documento_id' => 'required|exists:tipos_documento,id',
            'archivo' => 'required|file|max:10240',
            'fecha_vencimiento' => 'nullable|date',
            'observaciones' => 'nullable|string',
        ]);

        $cliente = Cliente::findOrFail($validated['cliente_id']);

        // Verificar que el cliente pertenece al vendedor
        if ($cliente->vendedor_id !== Auth::id()) {
            abort(403, 'No tienes acceso a este cliente.');
        }

        // Verificar si ya existe un documento del mismo tipo
        $tipoDocumento = TipoDocumento::findOrFail($validated['tipo_documento_id']);
        $documentoExistente = Documento::where('cliente_id', $validated['cliente_id'])
            ->where('tipo_documento_id', $validated['tipo_documento_id'])
            ->first();

        if ($documentoExistente) {
            // Eliminar documento anterior
            Storage::disk('public')->delete($documentoExistente->ruta_archivo);
            $documentoExistente->delete();
            
            // Registrar en historial
            HistorialCambio::create([
                'cliente_id' => $validated['cliente_id'],
                'usuario_id' => Auth::id(),
                'accion_realizada' => "Documento '{$tipoDocumento->nombre}' reemplazado.",
            ]);
        }

        // Validar y guardar archivo
        $resultado = $this->fileService->guardarDocumento(
            $request->file('archivo'),
            $cliente->rif
        );

        if (!$resultado['valid']) {
            return back()->withErrors(['archivo' => $resultado['error']]);
        }

        // Crear documento
        $documento = Documento::create([
            'cliente_id' => $validated['cliente_id'],
            'tipo_documento_id' => $validated['tipo_documento_id'],
            'nombre_archivo' => $resultado['nombre_archivo'],
            'ruta_archivo' => $resultado['ruta_archivo'],
            'fecha_vencimiento' => $validated['fecha_vencimiento'] ?? null,
            'observaciones' => $validated['observaciones'] ?? null,
            'creado_por_id' => Auth::id(),
        ]);

        // Registrar en historial si no fue reemplazo
        if (!$documentoExistente) {
            HistorialCambio::create([
                'cliente_id' => $validated['cliente_id'],
                'usuario_id' => Auth::id(),
                'accion_realizada' => "Documento '{$tipoDocumento->nombre}' subido.",
            ]);
        }

        return redirect()->back()
            ->with('success', $documentoExistente ? 'Documento reemplazado exitosamente.' : 'Documento subido exitosamente.');
    }

    public function destroy(Documento $documento)
    {
        // Verificar que el documento pertenece a un cliente del vendedor
        if ($documento->cliente->vendedor_id !== Auth::id()) {
            abort(403, 'No tienes acceso a este documento.');
        }

        $clienteId = $documento->cliente_id;
        $tipoDocumento = $documento->tipoDocumento;
        
        // Eliminar archivo físico
        Storage::disk('public')->delete($documento->ruta_archivo);
        
        $documento->delete();

        // Registrar en historial
        HistorialCambio::create([
            'cliente_id' => $clienteId,
            'usuario_id' => Auth::id(),
            'accion_realizada' => "Documento '{$tipoDocumento->nombre}' eliminado.",
        ]);

        return redirect()->back()
            ->with('success', 'Documento eliminado exitosamente.');
    }

    public function download(Documento $documento)
    {
        // Verificar acceso
        if ($documento->cliente->vendedor_id !== Auth::id()) {
            abort(403);
        }

        if (!Storage::disk('public')->exists($documento->ruta_archivo)) {
            abort(404, 'Archivo no encontrado');
        }

        return Storage::disk('public')->download(
            $documento->ruta_archivo,
            $documento->nombre_archivo
        );
    }
}

