<?php

namespace App\Services;

use App\Models\Cliente;
use App\Models\HistorialCambio;
use Illuminate\Support\Facades\DB;

class ClienteService
{
    public function crearCliente(array $data, int $usuarioId): Cliente
    {
        return DB::transaction(function () use ($data, $usuarioId) {
            $cliente = Cliente::create([
                'nombre_empresa' => $data['nombre_empresa'],
                'rif' => $data['rif'],
                'direccion_fiscal' => $data['direccion_fiscal'] ?? null,
                'zona' => $data['zona'] ?? null,
                'vendedor_id' => $data['vendedor_id'] ?? null,
                'notas' => $data['notas'] ?? null,
                'creado_por_id' => $usuarioId,
                'modificado_por_id' => $usuarioId,
            ]);

            // Registrar en historial
            HistorialCambio::create([
                'cliente_id' => $cliente->id,
                'usuario_id' => $usuarioId,
                'accion_realizada' => 'Cliente creado.',
            ]);

            return $cliente;
        });
    }

    public function actualizarCliente(Cliente $cliente, array $data, int $usuarioId): Cliente
    {
        return DB::transaction(function () use ($cliente, $data, $usuarioId) {
            $oldData = $cliente->toArray();
            
            $cliente->update([
                'nombre_empresa' => $data['nombre_empresa'],
                'rif' => $data['rif'],
                'direccion_fiscal' => $data['direccion_fiscal'] ?? null,
                'zona' => $data['zona'] ?? null,
                'vendedor_id' => $data['vendedor_id'] ?? null,
                'notas' => $data['notas'] ?? null,
                'modificado_por_id' => $usuarioId,
            ]);

            // Detectar cambios y registrar en historial
            $cambios = [];
            if ($oldData['nombre_empresa'] != $data['nombre_empresa']) {
                $cambios[] = 'nombre de la empresa';
            }
            if ($oldData['rif'] != $data['rif']) {
                $cambios[] = 'RIF';
            }
            if ($oldData['direccion_fiscal'] != ($data['direccion_fiscal'] ?? null)) {
                $cambios[] = 'dirección fiscal';
            }
            if ($oldData['zona'] != ($data['zona'] ?? null)) {
                $cambios[] = 'zona';
            }
            if ($oldData['vendedor_id'] != ($data['vendedor_id'] ?? null)) {
                $cambios[] = 'vendedor asignado';
            }
            if ($oldData['notas'] != ($data['notas'] ?? null)) {
                $cambios[] = 'notas';
            }

            if (!empty($cambios)) {
                HistorialCambio::create([
                    'cliente_id' => $cliente->id,
                    'usuario_id' => $usuarioId,
                    'accion_realizada' => 'Se modificó: ' . implode(', ', $cambios) . '.',
                ]);
            }

            return $cliente->fresh();
        });
    }
}

