<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class FileStorageService
{
    protected $allowedMimeTypes = [
        'application/pdf',
        'image/jpeg',
        'image/jpg',
        'image/png',
    ];

    protected $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png'];
    protected $maxSize = 10485760; // 10MB

    public function validarArchivo(UploadedFile $file): array
    {
        // Validar tamaño
        if ($file->getSize() > $this->maxSize) {
            return [
                'valid' => false,
                'error' => 'El archivo excede el tamaño máximo permitido (10MB).'
            ];
        }

        // Validar tipo MIME
        $mimeType = $file->getMimeType();
        if (!in_array($mimeType, $this->allowedMimeTypes)) {
            return [
                'valid' => false,
                'error' => 'Tipo de archivo no permitido. Solo se permiten PDF, JPG y PNG.'
            ];
        }

        // Validar extensión
        $extension = strtolower($file->getClientOriginalExtension());
        if (!in_array($extension, $this->allowedExtensions)) {
            return [
                'valid' => false,
                'error' => 'Extensión de archivo no permitida.'
            ];
        }

        // Verificar que el tipo MIME coincide con la extensión
        $mimeMap = [
            'pdf' => 'application/pdf',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png'
        ];

        if (isset($mimeMap[$extension]) && $mimeType !== $mimeMap[$extension]) {
            return [
                'valid' => false,
                'error' => 'El tipo de archivo no coincide con su extensión.'
            ];
        }

        return ['valid' => true, 'error' => null];
    }

    public function sanitizarNombreArchivo(string $filename): string
    {
        $filename = basename($filename);
        $filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filename);
        $filename = substr($filename, 0, 255);

        if (empty($filename)) {
            $filename = 'documento_' . time();
        }

        return $filename;
    }

    public function guardarDocumento(UploadedFile $file, string $rif): array
    {
        $validacion = $this->validarArchivo($file);
        if (!$validacion['valid']) {
            return $validacion;
        }

        $nombreOriginal = $this->sanitizarNombreArchivo($file->getClientOriginalName());
        $extension = $file->getClientOriginalExtension();
        $nombreArchivo = time() . '_' . $nombreOriginal;
        $rutaCarpeta = 'archivos/' . $rif;
        $rutaCompleta = $rutaCarpeta . '/' . $nombreArchivo;

        // Asegurar que el directorio existe
        Storage::disk('public')->makeDirectory($rutaCarpeta);

        // Guardar archivo
        $path = Storage::disk('public')->putFileAs(
            $rutaCarpeta,
            $file,
            $nombreArchivo
        );

        return [
            'valid' => true,
            'nombre_archivo' => $nombreArchivo,
            'ruta_archivo' => $path,
            'error' => null
        ];
    }

    public function eliminarDocumento(string $rutaArchivo): bool
    {
        return Storage::disk('public')->delete($rutaArchivo);
    }

    public function obtenerRutaArchivo(string $rif): string
    {
        $rif = preg_replace('/[^a-zA-Z0-9-]/', '', $rif);
        return 'archivos/' . $rif . '/';
    }
}

