<?php

namespace App\Services;

use App\Models\User;
use App\Models\PushSubscription;
use Minishlink\WebPush\WebPush;
use Minishlink\WebPush\Subscription;
use Illuminate\Support\Facades\Log;

class PushNotificationService
{
    protected $webPush;

    public function __construct()
    {
        try {
            $publicKey = config('services.vapid.public_key', env('VAPID_PUBLIC_KEY'));
            $privateKey = config('services.vapid.private_key', env('VAPID_PRIVATE_KEY'));
            
            // Solo inicializar WebPush si las claves están configuradas
            if ($publicKey && $privateKey) {
                $auth = [
                    'VAPID' => [
                        'subject' => config('services.vapid.subject', env('VAPID_SUBJECT', 'mailto:admin@example.com')),
                        'publicKey' => $publicKey,
                        'privateKey' => $privateKey,
                    ],
                ];
                $this->webPush = new WebPush($auth);
            } else {
                $this->webPush = null;
            }
        } catch (\Exception $e) {
            // Si hay error al inicializar, dejar webPush como null
            $this->webPush = null;
            Log::warning('PushNotificationService: No se pudo inicializar WebPush. ' . $e->getMessage());
        }
    }

    public function enviarNotificacion(User $user, string $titulo, string $mensaje, array $data = [])
    {
        if ($this->webPush === null) {
            Log::warning('PushNotificationService: WebPush no está inicializado. Verifica las claves VAPID.');
            return ['enviadas' => 0, 'fallidas' => 0];
        }

        $subscriptions = PushSubscription::where('user_id', $user->id)->get();

        if ($subscriptions->isEmpty()) {
            return ['enviadas' => 0, 'fallidas' => 0];
        }

        $payload = json_encode([
            'title' => $titulo,
            'body' => $mensaje,
            'icon' => asset('vendedor/assets/icons/icon-192x192.png'),
            'badge' => asset('vendedor/assets/icons/icon-72x72.png'),
            'data' => $data,
        ]);

        $enviadas = 0;
        $fallidas = 0;

        foreach ($subscriptions as $subscription) {
            try {
                $pushSubscription = Subscription::create([
                    'endpoint' => $subscription->endpoint,
                    'keys' => [
                        'p256dh' => $subscription->p256dh,
                        'auth' => $subscription->auth,
                    ],
                ]);

                $result = $this->webPush->sendOneNotification(
                    $pushSubscription,
                    $payload
                );

                if ($result->isSuccess()) {
                    $enviadas++;
                } else {
                    $fallidas++;
                    // Si la suscripción es inválida, eliminarla
                    if ($result->isSubscriptionExpired()) {
                        $subscription->delete();
                    }
                }
            } catch (\Exception $e) {
                Log::error('Error enviando push notification: ' . $e->getMessage());
                $fallidas++;
            }
        }

        // Enviar todas las notificaciones pendientes
        if ($this->webPush !== null) {
            $this->webPush->flush();
        }

        return [
            'enviadas' => $enviadas,
            'fallidas' => $fallidas,
        ];
    }

    public function enviarATodos(string $titulo, string $mensaje, array $data = [])
    {
        if ($this->webPush === null) {
            Log::warning('PushNotificationService: WebPush no está inicializado. Verifica las claves VAPID.');
            return ['enviadas' => 0, 'fallidas' => 0];
        }

        $usuarios = User::activos()->get();
        $totalEnviadas = 0;
        $totalFallidas = 0;

        foreach ($usuarios as $usuario) {
            $resultado = $this->enviarNotificacion($usuario, $titulo, $mensaje, $data);
            $totalEnviadas += $resultado['enviadas'];
            $totalFallidas += $resultado['fallidas'];
        }

        return [
            'enviadas' => $totalEnviadas,
            'fallidas' => $totalFallidas,
        ];
    }
}

