// PWA Manager - Gestión completa de funcionalidades PWA
// Maneja instalación, estado offline/online, y características PWA

class PWAManager {
    constructor() {
        this.deferredPrompt = null;
        this.isInstalled = false;
        this.init();
    }

    init() {
        // Detectar si está instalado como PWA
        this.checkIfInstalled();

        // Detectar estado de conexión
        this.setupConnectionListeners();

        // Setup install prompt
        this.setupInstallPrompt();

        // Setup periodic sync
        this.setupPeriodicSync();
    }

    checkIfInstalled() {
        // Verificar si está en modo standalone (instalado)
        if (window.matchMedia('(display-mode: standalone)').matches) {
            this.isInstalled = true;
            document.body.classList.add('pwa-installed');
            console.log('[PWAManager] PWA instalada');
        } else if (window.navigator.standalone === true) {
            // iOS Safari
            this.isInstalled = true;
            document.body.classList.add('pwa-installed');
            console.log('[PWAManager] PWA instalada (iOS)');
        } else {
            this.isInstalled = false;
            document.body.classList.remove('pwa-installed');
        }
    }

    setupConnectionListeners() {
        window.addEventListener('online', () => {
            console.log('[PWAManager] Conexión restaurada');
            this.updateConnectionStatus(true);
            this.showNotification('Conexión restaurada', 'success');
        });

        window.addEventListener('offline', () => {
            console.log('[PWAManager] Sin conexión');
            this.updateConnectionStatus(false);
            this.showNotification('Sin conexión. Algunas funciones pueden estar limitadas.', 'warning');
        });

        // Estado inicial
        this.updateConnectionStatus(navigator.onLine);
    }

    updateConnectionStatus(isOnline) {
        const indicator = document.getElementById('connection-status');
        if (indicator) {
            if (isOnline) {
                indicator.className = 'connection-status online';
                indicator.innerHTML = '<i class="fas fa-wifi"></i> En línea';
            } else {
                indicator.className = 'connection-status offline';
                indicator.innerHTML = '<i class="fas fa-wifi-slash"></i> Sin conexión';
            }
        }
    }

    setupInstallPrompt() {
        // Escuchar el evento beforeinstallprompt
        window.addEventListener('beforeinstallprompt', (e) => {
            // Prevenir el prompt automático
            e.preventDefault();
            // Guardar el evento para usarlo después
            this.deferredPrompt = e;
            console.log('[PWAManager] Install prompt disponible');
            
            // Mostrar botón de instalación
            this.showInstallButton();
        });

        // Detectar si la app fue instalada
        window.addEventListener('appinstalled', () => {
            console.log('[PWAManager] PWA instalada');
            this.isInstalled = true;
            this.deferredPrompt = null;
            this.hideInstallButton();
            this.showNotification('¡Aplicación instalada correctamente!', 'success');
        });
    }

    showInstallButton() {
        let installBtn = document.getElementById('pwa-install-btn');
        if (!installBtn) {
            installBtn = document.createElement('button');
            installBtn.id = 'pwa-install-btn';
            installBtn.className = 'pwa-install-btn';
            installBtn.innerHTML = '<i class="fas fa-download"></i> Instalar App';
            installBtn.onclick = () => this.install();
            
            // Agregar al header si existe
            const headerActions = document.querySelector('.header-actions');
            if (headerActions) {
                headerActions.insertBefore(installBtn, headerActions.firstChild);
            } else {
                document.body.appendChild(installBtn);
            }
        }
        installBtn.style.display = 'inline-flex';
    }

    hideInstallButton() {
        const installBtn = document.getElementById('pwa-install-btn');
        if (installBtn) {
            installBtn.style.display = 'none';
        }
    }

    async install() {
        if (!this.deferredPrompt) {
            this.showNotification('La aplicación ya está instalada o no se puede instalar', 'info');
            return;
        }

        // Mostrar el prompt de instalación
        this.deferredPrompt.prompt();

        // Esperar a que el usuario responda
        const { outcome } = await this.deferredPrompt.userChoice;
        console.log('[PWAManager] Usuario eligió:', outcome);

        if (outcome === 'accepted') {
            this.showNotification('Instalando aplicación...', 'info');
        } else {
            this.showNotification('Instalación cancelada', 'info');
        }

        // Limpiar el prompt
        this.deferredPrompt = null;
        this.hideInstallButton();
    }

    setupPeriodicSync() {
        // Registrar periodic sync si está disponible
        // Nota: periodicSync requiere permisos especiales y solo funciona en Chrome/Edge
        if ('serviceWorker' in navigator && 'periodicSync' in window.ServiceWorkerRegistration.prototype) {
            navigator.serviceWorker.ready.then(registration => {
                // Verificar si periodicSync está disponible
                if (registration.periodicSync) {
                    registration.periodicSync.register('sync-documents', {
                        minInterval: 24 * 60 * 60 * 1000 // 24 horas
                    }).then(() => {
                        console.log('[PWAManager] Periodic sync registrado');
                    }).catch(error => {
                        // No es crítico si falla, solo loguear
                        if (error.name !== 'NotAllowedError') {
                            console.warn('[PWAManager] Periodic sync no disponible:', error.message);
                        } else {
                            // NotAllowedError es normal si el usuario no ha dado permisos
                            console.log('[PWAManager] Periodic sync requiere permisos adicionales');
                        }
                    });
                }
            }).catch(error => {
                // Ignorar errores de periodic sync, no es crítico
                console.log('[PWAManager] Periodic sync no disponible en este navegador');
            });
        } else {
            // Periodic sync no está disponible, usar background sync normal
            console.log('[PWAManager] Usando background sync estándar');
        }
    }

    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `toast-notification ${type}`;
        notification.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'warning' ? 'exclamation-triangle' : 'info-circle'}"></i>
            <span>${message}</span>
        `;
        document.body.appendChild(notification);

        setTimeout(() => {
            notification.classList.add('show');
        }, 100);

        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => {
                notification.remove();
            }, 300);
        }, 3000);
    }
}

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    window.pwaManager = new PWAManager();
});
