// Service Worker para Portal de Vendedores PWA - Laravel
// Versión: 2.1.0 - Con soporte completo offline, push notifications y subdirectorio /documentos/

const CACHE_NAME = 'vendedores-pwa-v2.1.0';
const STATIC_CACHE_NAME = 'vendedores-static-v2.1.0';
const DYNAMIC_CACHE_NAME = 'vendedores-dynamic-v2.1.0';
const OFFLINE_QUEUE_DB = 'offline_queue_db';
const OFFLINE_QUEUE_STORE = 'documents';

// Archivos estáticos que se cachearán inmediatamente
// Nota: Las rutas son relativas al scope del Service Worker
const STATIC_FILES = [
  '/documentos/vendedor/',
  '/documentos/vendedor/dashboard',
  '/documentos/vendedor/login',
  '/documentos/vendedor/assets/css/style.css',
  '/documentos/assets/css/all.min.css',
  '/documentos/vendedor/manifest.json',
];

// URLs que siempre deben ir a la red (no cachear)
const NETWORK_FIRST_URLS = [
  '/documentos/api/',
  '/documentos/admin/',
  '/documentos/vendedor/documentos',
  '/documentos/vendedor/clientes',
  '/documentos/logout',
];

// Instalación del Service Worker
self.addEventListener('install', (event) => {
  console.log('[SW] Instalando Service Worker...');
  
  event.waitUntil(
    caches.open(STATIC_CACHE_NAME)
      .then((cache) => {
        console.log('[SW] Cacheando archivos estáticos...');
        return cache.addAll(STATIC_FILES);
      })
      .then(() => {
        console.log('[SW] Service Worker instalado correctamente');
        return self.skipWaiting();
      })
      .catch((error) => {
        console.error('[SW] Error durante la instalación:', error);
      })
  );
});

// Activación del Service Worker
self.addEventListener('activate', (event) => {
  console.log('[SW] Activando Service Worker...');
  
  event.waitUntil(
    caches.keys()
      .then((cacheNames) => {
        return Promise.all(
          cacheNames.map((cacheName) => {
            if (cacheName !== STATIC_CACHE_NAME && cacheName !== DYNAMIC_CACHE_NAME) {
              console.log('[SW] Eliminando cache obsoleto:', cacheName);
              return caches.delete(cacheName);
            }
          })
        );
      })
      .then(() => {
        console.log('[SW] Service Worker activado');
        return self.clients.claim();
      })
  );
});

// Interceptación de requests
self.addEventListener('fetch', (event) => {
  const request = event.request;
  const url = new URL(request.url);
  
  // Solo interceptar requests dentro del scope del Service Worker
  // El scope es /documentos/vendedor/, así que solo procesar URLs que contengan esto
  // O que sean del mismo origen y estén en /documentos/
  const isSameOrigin = url.origin === self.location.origin;
  const isInScope = url.pathname.includes('/documentos/vendedor/') || 
                    url.pathname.includes('/documentos/api/') || 
                    url.pathname.includes('/documentos/admin/') ||
                    (url.pathname.startsWith('/documentos/') && isSameOrigin);
  
  if (!isInScope) {
    // Si no está en nuestro scope, dejar que pase directamente
    return;
  }
  
  // NO interceptar requests de logout
  if (request.method === 'POST' && url.pathname.includes('logout')) {
    return; // Dejar que pase directamente a la red
  }
  
  // Manejar POST requests (subida de documentos)
  if (request.method === 'POST' && url.pathname.includes('documentos')) {
    event.respondWith(handlePostRequest(event));
    return;
  }
  
  // Solo interceptar requests GET
  if (request.method !== 'GET') {
    return;
  }
  
  // Estrategia: Network First para APIs y acciones
  if (isNetworkFirstRequest(url)) {
    event.respondWith(networkFirstStrategy(request));
    return;
  }
  
  // Estrategia: Cache First para archivos estáticos
  if (isStaticFile(url)) {
    event.respondWith(cacheFirstStrategy(request));
    return;
  }
  
  // Estrategia: Stale While Revalidate para páginas
  event.respondWith(staleWhileRevalidateStrategy(request));
});

// Funciones de estrategia de caché
function networkFirstStrategy(request) {
  return fetch(request)
    .then((response) => {
      // Solo cachear respuestas exitosas y que sean cacheables
      if (response.ok && response.status === 200) {
        // No cachear respuestas de API que no sean estáticas
        if (!request.url.includes('/api/')) {
          const responseClone = response.clone();
          caches.open(DYNAMIC_CACHE_NAME).then((cache) => {
            cache.put(request, responseClone).catch(err => {
              console.warn('[SW] Error cacheando respuesta:', err);
            });
          });
        }
      }
      return response;
    })
    .catch((error) => {
      console.warn('[SW] Error de red, intentando cache:', error);
      return caches.match(request).then(cachedResponse => {
        if (cachedResponse) {
          return cachedResponse;
        }
        // Si no hay cache y es una página, devolver página offline
        if (request.mode === 'navigate') {
          return caches.match('/documentos/vendedor/offline.html') || new Response('Sin conexión', {
            status: 503,
            statusText: 'Service Unavailable',
            headers: { 'Content-Type': 'text/html' }
          });
        }
        throw error;
      });
    });
}

function cacheFirstStrategy(request) {
  return caches.match(request)
    .then((response) => {
      return response || fetch(request).then((fetchResponse) => {
        if (fetchResponse.ok) {
          const responseClone = fetchResponse.clone();
          caches.open(STATIC_CACHE_NAME).then((cache) => {
            cache.put(request, responseClone);
          });
        }
        return fetchResponse;
      });
    });
}

function staleWhileRevalidateStrategy(request) {
  const cachePromise = caches.open(DYNAMIC_CACHE_NAME);
  const fetchPromise = fetch(request).then((response) => {
    if (response.ok) {
      const responseClone = response.clone();
      cachePromise.then((cache) => {
        cache.put(request, responseClone);
      });
    }
    return response;
  });
  
  return cachePromise.then((cache) => {
    return cache.match(request) || fetchPromise;
  });
}

function isNetworkFirstRequest(url) {
  return NETWORK_FIRST_URLS.some(pattern => url.pathname.includes(pattern));
}

function isStaticFile(url) {
  return /\.(css|js|png|jpg|jpeg|gif|svg|woff|woff2|ttf|eot)$/i.test(url.pathname);
}

// Manejar POST requests offline
async function handlePostRequest(event) {
  const request = event.request;
  
  try {
    // Intentar enviar a la red primero
    const response = await fetch(request.clone());
    
    // Si la respuesta es exitosa, retornarla
    if (response.ok) {
      return response;
    }
    
    // Si la respuesta no es exitosa pero hay conexión, retornar el error
    if (response.status >= 400 && response.status < 500) {
      return response; // Errores del servidor (4xx) se retornan directamente
    }
    
    // Para errores 5xx o de red, intentar guardar offline
    throw new Error('Network error');
  } catch (error) {
    console.log('[SW] Sin conexión, guardando en cola offline');
    
    // Solo guardar en cola si es una subida de documento
    if (request.url.includes('documentos') && request.method === 'POST') {
      try {
        const formData = await request.clone().formData();
        await saveToOfflineQueue(request.url, request.method, formData);
        
        return new Response(JSON.stringify({
          success: false,
          offline: true,
          message: 'Documento guardado para sincronizar cuando vuelva la conexión'
        }), {
          status: 202,
          statusText: 'Accepted',
          headers: { 
            'Content-Type': 'application/json',
            'X-Offline-Queue': 'true'
          }
        });
      } catch (queueError) {
        console.error('[SW] Error guardando en cola offline:', queueError);
        return new Response(JSON.stringify({
          success: false,
          error: 'No se pudo guardar el documento. Intenta nuevamente cuando tengas conexión.'
        }), {
          status: 503,
          headers: { 'Content-Type': 'application/json' }
        });
      }
    }
    
    // Para otros POST requests, retornar error
    return new Response(JSON.stringify({
      success: false,
      error: 'Sin conexión. Intenta nuevamente más tarde.'
    }), {
      status: 503,
      headers: { 'Content-Type': 'application/json' }
    });
  }
}

// Guardar en cola offline
async function saveToOfflineQueue(url, method, formData) {
  const db = await openDB();
  if (!db) return;
  
  const transaction = db.transaction([OFFLINE_QUEUE_STORE], 'readwrite');
  const store = transaction.objectStore(OFFLINE_QUEUE_STORE);
  
  const data = {
    url: url,
    method: method,
    formData: await serializeFormData(formData),
    timestamp: Date.now()
  };
  
  store.add(data);
}

// Abrir IndexedDB
function openDB() {
  return new Promise((resolve) => {
    const request = indexedDB.open(OFFLINE_QUEUE_DB, 1);
    
    request.onerror = () => resolve(null);
    request.onsuccess = () => resolve(request.result);
    
    request.onupgradeneeded = (event) => {
      const db = event.target.result;
      if (!db.objectStoreNames.contains(OFFLINE_QUEUE_STORE)) {
        const store = db.createObjectStore(OFFLINE_QUEUE_STORE, { keyPath: 'id', autoIncrement: true });
        store.createIndex('timestamp', 'timestamp', { unique: false });
      }
    };
  });
}

// Serializar FormData
async function serializeFormData(formData) {
  const data = {};
  for (const [key, value] of formData.entries()) {
    if (value instanceof File) {
      const arrayBuffer = await value.arrayBuffer();
      data[key] = {
        type: 'file',
        name: value.name,
        type_mime: value.type,
        data: Array.from(new Uint8Array(arrayBuffer))
      };
    } else {
      data[key] = value;
    }
  }
  return data;
}

// Push Notifications
self.addEventListener('push', (event) => {
  const data = event.data ? event.data.json() : {};
  const title = data.title || 'Notificación';
  const options = {
    body: data.body || '',
    icon: data.icon || '/documentos/vendedor/assets/icons/icon-192x192.png',
    badge: data.badge || '/documentos/vendedor/assets/icons/icon-72x72.png',
    data: data.data || {},
    tag: data.tag || 'default',
    requireInteraction: data.requireInteraction || false,
  };
  
  event.waitUntil(
    self.registration.showNotification(title, options)
  );
});

// Click en notificación
self.addEventListener('notificationclick', (event) => {
  event.notification.close();
  
  const notificationData = event.notification.data || {};
  const urlToOpen = notificationData.url || '/documentos/vendedor/dashboard';
  
  event.waitUntil(
    clients.matchAll({ type: 'window', includeUncontrolled: true })
      .then((clientList) => {
        for (const client of clientList) {
          if (client.url.includes('/documentos/vendedor/') && 'focus' in client) {
            return client.focus();
          }
        }
        if (clients.openWindow) {
          return clients.openWindow(urlToOpen);
        }
      })
  );
});

// Background Sync
self.addEventListener('sync', (event) => {
  if (event.tag === 'sync-offline-queue') {
    event.waitUntil(syncOfflineQueue());
  }
});

// Sincronizar cola offline
async function syncOfflineQueue() {
  const db = await openDB();
  if (!db) return;
  
  const transaction = db.transaction([OFFLINE_QUEUE_STORE], 'readonly');
  const store = transaction.objectStore(OFFLINE_QUEUE_STORE);
  const allRequests = store.getAll();
  
  return new Promise((resolve) => {
    allRequests.onsuccess = async () => {
      const items = allRequests.result;
      console.log(`[SW] Sincronizando ${items.length} documentos...`);
      
      for (const item of items) {
        try {
          const formData = new FormData();
          for (const [key, value] of Object.entries(item.formData)) {
            if (value && value.type === 'file') {
              const blob = new Blob([value.data], { type: value.type_mime });
              const file = new File([blob], value.name, { type: value.type_mime });
              formData.append(key, file);
            } else {
              formData.append(key, value);
            }
          }
          
          const response = await fetch(item.url, {
            method: item.method,
            body: formData
          });
          
          if (response.ok) {
            const deleteTransaction = db.transaction([OFFLINE_QUEUE_STORE], 'readwrite');
            const deleteStore = deleteTransaction.objectStore(OFFLINE_QUEUE_STORE);
            deleteStore.delete(item.id);
            console.log(`[SW] Documento ${item.id} sincronizado exitosamente`);
          }
        } catch (error) {
          console.error(`[SW] Error al sincronizar documento ${item.id}:`, error);
        }
      }
      
      resolve();
    };
  });
}

