<?php

/**
 * Script simple para generar claves VAPID usando OpenSSL
 * Uso: php scripts/generate-vapid-simple.php
 */

echo "Generando claves VAPID usando OpenSSL...\n\n";

if (!extension_loaded('openssl')) {
    die("❌ Error: OpenSSL no está disponible. Instala la extensión OpenSSL de PHP.\n");
}

try {
    // Configuración para clave EC P-256
    $config = [
        'digest_alg' => 'sha256',
        'private_key_bits' => 256,
        'private_key_type' => OPENSSL_KEYTYPE_EC,
        'curve_name' => 'prime256v1',
    ];
    
    // Generar par de claves
    $privateKeyResource = openssl_pkey_new($config);
    
    if (!$privateKeyResource) {
        $error = openssl_error_string();
        throw new Exception("Error generando clave: " . ($error ?: 'Error desconocido'));
    }
    
    // Exportar clave privada
    openssl_pkey_export($privateKeyResource, $privateKeyPEM);
    
    // Obtener detalles de la clave pública
    $keyDetails = openssl_pkey_get_details($privateKeyResource);
    
    if (!isset($keyDetails['ec'])) {
        throw new Exception("No se pudo obtener detalles de la clave EC");
    }
    
    // Extraer coordenadas X e Y
    $x = $keyDetails['ec']['x'];
    $y = $keyDetails['ec']['y'];
    
    // Construir clave pública en formato no comprimido (0x04 + X + Y)
    $publicKeyBytes = "\x04" . $x . $y;
    
    // Convertir a Base64 URL-safe
    $publicKey = base64UrlEncode($publicKeyBytes);
    
    // Extraer clave privada (d)
    $privateKeyBytes = $keyDetails['ec']['d'];
    $privateKey = base64UrlEncode($privateKeyBytes);
    
    // Mostrar resultados
    echo "═══════════════════════════════════════════════════════════\n";
    echo "✅ Claves VAPID generadas exitosamente!\n";
    echo "═══════════════════════════════════════════════════════════\n\n";
    
    echo "📋 Copia estas líneas y agrégalas a tu archivo .env:\n\n";
    echo "VAPID_SUBJECT=mailto:admin@example.com\n";
    echo "VAPID_PUBLIC_KEY=" . $publicKey . "\n";
    echo "VAPID_PRIVATE_KEY=" . $privateKey . "\n\n";
    
    echo "⚠️  IMPORTANTE:\n";
    echo "   - Reemplaza 'admin@example.com' con tu email real\n";
    echo "   - Nunca compartas tu clave privada públicamente\n";
    echo "   - Guarda estas claves de forma segura\n\n";
    
    // Intentar escribir directamente al .env si existe
    $envPath = __DIR__ . '/../.env';
    if (file_exists($envPath)) {
        echo "¿Deseas agregar estas claves automáticamente al .env? (s/n): ";
        $handle = fopen("php://stdin", "r");
        $line = trim(fgets($handle));
        fclose($handle);
        
        if (strtolower($line) === 's' || strtolower($line) === 'y' || strtolower($line) === 'si') {
            $envContent = file_get_contents($envPath);
            
            // Remover claves VAPID existentes si las hay
            $envContent = preg_replace('/^VAPID_SUBJECT=.*$/m', '', $envContent);
            $envContent = preg_replace('/^VAPID_PUBLIC_KEY=.*$/m', '', $envContent);
            $envContent = preg_replace('/^VAPID_PRIVATE_KEY=.*$/m', '', $envContent);
            
            // Agregar nuevas claves
            $envContent .= "\n# VAPID Keys para Push Notifications\n";
            $envContent .= "VAPID_SUBJECT=mailto:admin@example.com\n";
            $envContent .= "VAPID_PUBLIC_KEY=" . $publicKey . "\n";
            $envContent .= "VAPID_PRIVATE_KEY=" . $privateKey . "\n";
            
            file_put_contents($envPath, $envContent);
            echo "\n✅ Claves agregadas al archivo .env\n";
            echo "⚠️  Recuerda cambiar 'admin@example.com' por tu email real\n\n";
        }
    }
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n\n";
    echo "Alternativas:\n";
    echo "1. Usar herramienta online: https://www.stephane-quantin.com/en/tools/generators/vapid-keys\n";
    echo "2. Usar Node.js: npm install -g web-push && web-push generate-vapid-keys\n";
    exit(1);
}

function base64UrlEncode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

