<?php

/**
 * Script independiente para generar claves VAPID
 * Uso: php scripts/generate-vapid.php
 */

require __DIR__ . '/../vendor/autoload.php';

echo "Generando claves VAPID...\n\n";

try {
    // Intentar usar minishlink/web-push
    if (class_exists('\Minishlink\WebPush\VAPID')) {
        $keys = \Minishlink\WebPush\VAPID::createVapidKeys();
        displayKeys($keys);
        exit(0);
    }
    
    // Método alternativo usando OpenSSL
    if (!extension_loaded('openssl')) {
        throw new Exception('OpenSSL no está disponible');
    }
    
    echo "Usando método alternativo con OpenSSL...\n\n";
    
    $config = [
        'digest_alg' => 'sha256',
        'private_key_bits' => 2048,
        'private_key_type' => OPENSSL_KEYTYPE_EC,
        'curve_name' => 'prime256v1',
    ];
    
    $resource = openssl_pkey_new($config);
    if (!$resource) {
        throw new Exception('Error generando clave: ' . openssl_error_string());
    }
    
    openssl_pkey_export($resource, $privateKeyPEM);
    $publicKeyDetails = openssl_pkey_get_details($resource);
    
    // Extraer coordenadas
    $x = bin2hex($publicKeyDetails['ec']['x']);
    $y = bin2hex($publicKeyDetails['ec']['y']);
    
    // Convertir a Base64 URL-safe
    $publicKey = base64UrlEncode(hex2bin('04' . $x . $y));
    $privateKey = base64UrlEncode($publicKeyDetails['ec']['d']);
    
    displayKeys([
        'publicKey' => $publicKey,
        'privateKey' => $privateKey
    ]);
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n\n";
    echo "Alternativas:\n";
    echo "1. npm install -g web-push && web-push generate-vapid-keys\n";
    echo "2. Visita: https://www.stephane-quantin.com/en/tools/generators/vapid-keys\n";
    exit(1);
}

function base64UrlEncode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

function displayKeys($keys) {
    echo "═══════════════════════════════════════════════════════════\n";
    echo "✅ Claves VAPID generadas exitosamente!\n";
    echo "═══════════════════════════════════════════════════════════\n\n";
    echo "Agrega estas líneas a tu archivo .env:\n\n";
    echo "VAPID_SUBJECT=mailto:tu-email@ejemplo.com\n";
    echo "VAPID_PUBLIC_KEY=" . $keys['publicKey'] . "\n";
    echo "VAPID_PRIVATE_KEY=" . $keys['privateKey'] . "\n\n";
    echo "⚠️  IMPORTANTE:\n";
    echo "   - Reemplaza 'tu-email@ejemplo.com' con un email válido\n";
    echo "   - Nunca compartas tu clave privada\n";
    echo "   - Guarda estas claves de forma segura\n\n";
}

